#' Allocation probabilities for FLWE and FLGI procedures
#'
#' Calculation of allocation probabilities for FLWE and FLGI procedures
#' using Monte Carlo simulations.
#' @name dbtteppmx
#' @param Yin Response data as (n x p0) matrix of p0-dimensional responses for n experimental units.
#' @param p0 dimension of the response vector for each data record
#' @param p1 number of continuous covariates (can be 0 for none)
#' @param p2 number of categorical covariates.
#' @param q2 vector of length p2; number of levels for each categorical covariate.
#' @param p4 number of count covariates.
#' @param censoring  indicator for censoring
#' @param cov.Vj1  vector of length p1, variance parameters for the similarity function for continuous covariates.
#' @param cov.mean1 vector of length p1, prior mean of the location parameter for the similarity func- tion for continuous covariates.
#' @param cov.B1   par of auxiliary for cont cov 
#' @param cov.std1 indicator for standardizing continuous covariates to sample mean 0 and standard deviation 1.
#' @param cov.pi2  array with p2 rows and q2[j] columns in row j. The j-th row defines the Dirichlet parameters for the similarity function for the j-th categorical covariate.
#' @param cov.a4 vector of length p4, defines the shape parameter of the Gamma prior in the similarity function for count covariates.
#' @param cov.b4 vector of length p4, defines the scale parameter of the Gamma prior in the simi- larity function for count covariates.
#' @param n.iter number MCMC iterations
#' @param n.discard initial transient
#' @param n.reinit reinitialize every n.reinit iterations (not normally used
#' @param n.batch save imputed parameters every n.batch iterations
#' @param n.predupdate save posterior predictive summaries every n.predupdate iterations
#' @param m.prior indicator for resampling (1) versus fixing (0) the mean of the base measure, m
#' @param B.prior indicator for resampling (1) versus fixing (0) the variance-covariance matrix of the base measure, B
#' @param S.prior indicator for resampling (1) versus fixing (0) the sampling variance-covariance matrix
#' @param verbose level of comments
#' @param py indicator for evaluating posterior predictive p(y | data) for a future observation
#' @param iy The coordinate of the response for posterior predictive inference
#' @param ny Size of the grid on which to evaluate posterior predictive inference
#' @param ygrid vector of length 2, lower and upper bound of the grid on which to evaluate posterior predictive inference
#' @param nx  number of covariate combinations for which to carry out the posterior predictive inference
#' @param xgrid  actual cov-combinations
#' @param s degrees of freedom for the inverse Wishart prior on S
#' @param S.init initial value for the kernel covariance matrix S
#' @param qq degrees of freedom for the inverse Wishart base measure
#' @param R expectation of the inverse Wishart base measure 
#' @param B.scale The initial value for the covariance matrix B of the base measure G0(mu) = N (m, B) is set to B scale S init.
#' @param cc degrees of freedom of the inverse Wishart hyperprior for B
#' @param m.init initial value for the mean m of the base measure G0(mu) = N(m,B)
#' @param a hyperprior mean for m
#' @param A.inv inverse of the hyperprior covariance matrix for m
#' @param alpha initial value for the total mass parameter alpha
#' @param a0 hyperprior parameters for prior on total mass paramter alpha
#' @param b0 hyperprior parameters for prior on total mass paramter alpha
#' @param k0  initial number of distinct clusters
#' @param postMCMC indicator for stage2
#' @param sigma1 initial value of varaince for Metropolis-Hastings algorithm
#' @param sigma2 initial value of varaince for Metropolis-Hastings algorithm
#' @param seeds index for naming for output data file
#' @export
#' @examples
#' dbtte_analysis<-dbtteppmx(data,n.discard=1000,n.iter=3000,censoring=1,
#'                  p0=2,                           # 1-dim response
#'                  p1=1,p2=2,p4=0,                 # 1 cont and 2 categorical cov's
#'                 B.prior=1,S.prior=1,cov.std1=1,n.batch=5,cov.a4=1, cov.b4=0.25, postMCMC=0,n.reinit=0,ny=150,iy=0)
#'dbtteppmx::convolve_from_c(dbtte_analysis)
require(dplyr)
require(parmsurvfit)
require(Rcpp)
dbtteppmx <-  function(Yin=NULL,p0=1,p1=NULL,p2=NULL,q2=NULL,p4=0,
                      censoring=0,   # indicator for censoring of response
                                     ## only for p0=1
                      cov.dahl=0,    # indicator for using Dahl model
                      cov.Vj1=NULL,  # variance of aux model for cont cov
                      cov.mean1=NULL,# prior mean & var for location 
                      cov.B1=NULL,   #    par of auxiliary for cont cov
                      cov.std1=1,    # indicator for standardizing cont covs
                      cov.pi2=NULL,
                      cov.a4=NULL,
                      cov.b4=NULL,
                      n.iter=2000,    # n MCMC iterations
                      n.discard=1000,  # initial transient
                      n.reinit=10000, # reinitialize
                      n.batch=50,     # batch size to thin out
                      n.predupdate=100, # batch size to update predictives
                      n.printallpars=1000, # batch size to print all pars 
                      m.prior=0,      # resample m
                      B.prior=0,      # resample B
                      S.prior=0,      # resample B
                      verbose=1,      # level of comments
                      py=1,           # marginal posterior predictive
                      iy=NULL,        # index for y
                      ny=50,          # grid size for y
                      ygrid=NULL,     # (min,max) of ygrid
                      nx=NULL,        # number of cov-combinations
                      xgrid=NULL,     # actual cov-combinations
                      s=15,           # d.f. for prior on S
                      S.init=NULL,    # initial value and prior mean for S
                      qq=5,           # d.f. for prior on R
                      R=NULL,         # hyperprior parameter R
                      B.scale=100,    # initial value for B = B.scale*S.init,
                      cc=5,           # d.f. par c
                      m.init=NULL,    # initial values for m
                      a=NULL,         # hyperpar a
                      A.inv=NULL,     # hyperpar A
                      alpha=1,        # initial value for total mass alpha
                      a0=1,
                      b0=1,           # hyperpr pars for alpha ~ IG(a0/2,b0/2)
                      k0=NULL,postMCMC=0, # Binary indicator of Stage 1: 0 ; Stage 2 : 1
                      sigma1=0.1,sigma2=0.01, # initial value of varaince for Metropolis-Hastings algorithm
                      seeds=0,simu=0,tr_p1=1,tr_p2=2)        # simulation seeds for random variables
                      
  { # read in data
    if (is.null(Yin)){
      cat("\n *** Error: need data matrix or file Yin.\n")
      return(-1)
    }
    if(is.character(Yin)){
      if (!file.exists(Yin)){ ## reading does not work
        cat("\n *** Error: could not read data from file",Yin,
            "\n Hint: need to include full path.\n")
        return(-1)
      }
      Y <- read.table(Yin,header=T)
      if (verbose>0){
        cat("Read ", nrow(Y), " rows and ", ncol(Y), " columns",
            "from data file ", Yin,"\n")
        cat("Variable (column) names are: ",names(Y),"\n")
      }
    } else { # not char
      if ( (!is.matrix(Yin)) & (!is.data.frame(Yin))){
        cat("\n *** Error: need Yin = filename or matrix or data frame.\n")
        return(-1)
      }
      Y <- Yin
    }
    if (is.null(p0) | is.null(p1) | is.null(p2) | is.null(p4)){
        cat("\n *** Error: need to set p0,p1,p2,p4.\n")
        return(-1)
      }
    if(!censoring)
      p <- ncol(Y)
    else if(censoring && p0==1)
      p<- ncol(Y)-1
    else
       p<- ncol(Y)-2
   
    n <- nrow(Y)
    q <- apply(Y,2,n.unique) # will use later..
    if (p0+p1+p2+p4 != p){
      cat("\n *** Error: n columns != p0+p1+p2+p4.\n")
      return(-1)
    }
    ## continuous covariates: parameters and standardize
    if (p1>0){
      idx1 <- (p0+1):(p0+p1)
      if (is.null(cov.Vj1))
        cov.Vj1 <- rep(0.25,p1)
      if (is.null(cov.mean1))
        cov.mean1 <- rep(0,p1)
      if (is.null(cov.B1))
        cov.B1 <- rep(1,p1)
      if ((length(cov.Vj1)!=p1) | (length(cov.mean1)!=p1) |
          (length(cov.B1)!=p1 ) ){
        cat("\n *** Error: cov.Vj1, mean1 and B1 need to be of length p1.\n")
        return(-1)
      }

      if (any(q[idx1] <= 3))
        cat("\n *** Warning: check that columns", (p0+1), "through", (p0+p1),"\n",
            "    are really continous covariates. Only",q[idx1],
            " unique values.\n")
      if (cov.std1==1){
        Y1 <- as.matrix( Y[,idx1] ) # as.matrix in case p1=1
        v1 <- apply(Y1,2,var)       # empirical var of cont cov's
        s1 <- sqrt(v1)
        m1 <- apply(Y1,2,mean)      # empirical mean
        y1 <- t( (t(Y1) - m1)/s1 )  # standardized continuous vars
        Y[,idx1] <- y1               # substitute Y
      }
    }
    ## categorical covariates
    if (p2>0){
      idx2 <- (p0+p1+1):(p0+p1+p2)
      q2 <- q[idx2]
      if (any(q2 > 10))
        cat("\n *** Warning: check that cols (p0+p1+1...p0+p1+p2)",
            "\n     are really categorical. ")
      cat("\n Finding ", q[idx2],
          " unique values for categorical covariates in cols",
          p0+p1+1,"--", p0+p1+p2,"\n")
      if (is.null(cov.pi2)){
        cov.pi2 <- vector("list",p2)
        for(j in 1:p2)
          cov.pi2[[j]] <- rep(0.1,q2[j])
      }
      if (length(unlist(cov.pi2)) != sum(q2)){
        cat("\n *** Error: need length(pi2[j]) = # levels of j-th ",
            "\n     categorical covariate.\n")
        return(-1)
      }
    }
    ## initial clustering
    if(is.null(k0)){
      k0 <- ifelse(p2>0,0,n)
      cat(" setting default for k0=",k0,"\n")
    }
    if (!is.element(k0,c(0,1,n))){
      cat("\n *** Error: need k0= 0, 1 or n. \n")
      return(-1)
    }
    if (k0 == 0){ # initialize by cross tabulation of categoricals
      Y2 <- Y[,idx2]
      q2 <- q[idx2]
      member0 <- cross.tab(Y2,q2)
      k0 <- length(unique(member0))
      cat("\n Using cross-tabulation by categoricals as member0.\n")
    } else
      member0 <- 0 # dummy
    ## response variable
    if (!is.element(censoring,c(0,1))){
      cat("\n *** Error: need censoring=0 or 1.\n")
      return(-1)
    }
 
    idx0 <- 1:p0
    Y0 <- as.matrix(Y[,idx0]) # need as.matrix for p0=1
    mhat <- apply(Y0,2,mean)
    V <- var(Y0)
    if(p0==1)
      {
        v <- as.matrix( diag(V)  )  # as.matrix in case p0=1
      }
    else{
        v <- as.matrix( V ) 
    }
    if(is.null(S.init)){
      S.init <- 0.25*V
     
       
      cat(" setting default for S.init = 1/4*var(Y).\n")
    }
    if(is.null(R)){
      R <- 0.25*v
      cat(" setting default for R = 1/4*diag(v).\n")
    }
    if(is.null(m.init)){
      m.init <- mhat
      cat(" setting default for m.init = mean(Y).\n")
    }
    if(is.null(A.inv)){
      A.inv <- (1/v)
      cat(" setting default for A = diag(v).\n")
    }
    if(is.null(a)){
      a <- mhat
      cat(" setting default for a = mean(Y).\n")
    }
    ## py-init
    if (!is.element(py,c(0,1))){
      cat("\n *** Error: need py=0 or 1.\n")
      return(-1)
    }
    if(py==1){
      if(is.null(iy)){
        iy <- 0
        cat("\n Computing marginals for response in column",iy+1,"\n")
      }
      if (iy >= p0){
        cat("\n *** Error: iy needs to indicate one of the first p0=",p0,
            "\n     columns (CAREFUL: iy starts counting at 0 for col. 1.\n")
        return(-1)
      }
      
      yy <- Y[,iy+1]

      if(p0>1){
         yy2<-Y[,iy+2]
      }
     
      
      if(is.null(ygrid)){
        ygrid <- range(yy)
        if(p0>1){
          ygrid2 <- range(yy2)
        }
      }
      if((ygrid[1]<min(yy)) | (ygrid[2]>max(yy))){
        cat("\n *** Error: can not set range of ygrid beyond range of data.\n")
        return(-1)
      }
      if(is.null(nx)){
        if(!is.null(xgrid))
          nx <- nrow(xgrid)
        else{
          if(p1+p2==0){
            cat("\n Using first 5 cov vectors as xgrid.\n")
            xgrid <- Y[1:5,-(1:p0)] # simply use first 5 design vectors
            nx <- 5
          } else { # use cross tab of all categoricals & (hi,med,lo) cont
            cat("\n Using cross tabulation of categoricals for xgrid.\n")
            p12 <- p1+p2
            q12 <- c(rep(3,p1),q2)
            nx <- prod(q12)
            xgrid <- cross.tab2(q12,nx,p1)
            if (p4>0){
              m4 <- round( apply(Y[,(p-p4+1):p4],2,mean) )
              for(j in 1:p4)
                xgrid <- cbind(xgrid,m4[j])
            }
          } # xgrid
        } # is.null xgrid
      }
    } # py==1

    ## MCMC tuning pars
    if (n.discard >= n.iter){
        cat("\n *** Error: n.iter <= n.discard. \n")
        return(-1)
      }
    if (n.batch >= n.iter/2){
      cat("\n *** Error: n.iter/2 <= n.batch. \n")
      return(-1)
    }
    ## warnings
    if (cc < p0+2)
      cat(" *** Warning: should use c > p+2 for good mixing MCMC.\n")
    if (qq < p0+2)
      cat(" *** Warning: should use q > p+2 for good mixing MCMC.\n")
    if (s < p0+2)
      cat(" *** Warning: should use s > p+2 for good mixing MCMC.\n")

    ##### File names setting
    if(seeds==0)
    {
       v_name <- paste("V", ".mdp", sep = "")
      muname <- paste("mu", ".mdp", sep = "")
      mj1name <- paste("mj1", ".mdp", sep = "")
      pij2name<- paste("pij2",  ".mdp", sep = "")
      aj4name <- paste("aj4", ".mdp", sep = "")
      survaname <-paste("surv-a", ".mdp", sep = "")
      survbname <-paste("surv-b", ".mdp", sep = "")
      lambdaname<-paste("lambda", ".mdp", sep = "")
      copulaname<-paste("copula",  ".mdp", sep = "")

      membername<-paste("member",  ".mdp", sep = "")
      parname <- paste("par",  ".mdp", sep = "")
      Bname <- paste("B",  ".mdp", sep = "")

    }else{
         v_name <- paste("V_", seeds, ".mdp", sep = "")
      muname <- paste("mu_", seeds, ".mdp", sep = "")
      mj1name <- paste("mj1_", seeds, ".mdp", sep = "")
      pij2name<- paste("pij2_", seeds, ".mdp", sep = "")
      aj4name <- paste("aj4_", seeds, ".mdp", sep = "")
      survaname <-paste("surva_", seeds, ".mdp", sep = "")
      survbname <-paste("survb_", seeds, ".mdp", sep = "")
      lambdaname<-paste("lambda_", seeds, ".mdp", sep = "")
      copulaname<-paste("copula_", seeds, ".mdp", sep = "")

      membername<-paste("member_", seeds, ".mdp", sep = "")
      parname <- paste("par_", seeds, ".mdp", sep = "")
      Bname <- paste("B_", seeds, ".mdp", sep = "")

    }
      #### Read all previous parameters from Stage 1 for Stage2
    if (postMCMC >= 1){
      #read membership

      # lsfile<-read.table("/Users/depcts/Desktop/coppmxcpp/lsmember.mdp") 
      # ### read the rows and column can be possible.
      # sourceCpp("least_square.cpp")
      # v<-lsclu("/Users/depcts/Desktop/coppmxcpp/lsmember.mdp",nrow(lsfile),ncol(lsfile)) 
      # num_class1=length(table(v))

       myData = read.delim(membername, header = FALSE)
      
      num_class <- read.table(membername)[,2]
      

      ### by frenquency
      num_class1 <- find_freq(num_class)
      tem_index=which(num_class==num_class1)

  
      if(postMCMC==1){
          mem_index=tem_index[length(tem_index)]
      }else if(postMCMC==2){
        mem_index = sample(tem_index, 1)     #### Random select one
      }
      
      mymember_k= myData[1]
      mymember_k<-(as.character(unlist(mymember_k)))
      mymember_k<-strsplit(mymember_k, "  ")
      mymember_k<-mymember_k[mem_index]
      
       ###  mymember_k<-mymember_k[length(mymember_k)]
      mymember_k<-(as.character(unlist(mymember_k)))
      mymember_k<-strsplit(mymember_k, "  ")
      iter_and_nc<-as.numeric( unlist(mymember_k))
      iteration_index<-iter_and_nc[1]
      k0=as.integer(num_class1)
      
       ### each subject membership
      member1= myData[2]
      member1<-(as.character(unlist(member1)))
      member1<-strsplit(member1, "  ")
      member1<-member1[mem_index]
      member1<-(as.character(unlist(member1)))
      
      member0<-unlist(member1[member1 !=""])
       ### member1 = member1[-which(sapply(member1, is.null))]
      
      
      
      #read alpha-init
      myalpha = read.delim(parname, header = FALSE)
      myalpha1<-(as.character(unlist(myalpha)))
      myalpha2<-strsplit(myalpha1, "  ")
     # myalpha2=myalpha2[length(myalpha2)]
      myalpha2=myalpha2[mem_index-1]  # be careful. from c, it is from 0 to n-1
     
      myalpha2=unlist(myalpha2)
      alpha=myalpha2[2]
      
     
      #eta
      eta_preset=myalpha2[3]  #is drawn from alpha
    
      #S.init
      if(p0>1){
            myB = read.csv(Bname, header = FALSE)
      }
      else{
             myB = read.table(Bname, header = FALSE)
              B_index=match(iteration_index-1,myB)#notice the iteration index in B. because it is from 0 to n-1
              myB1=myB[B_index+1]
              
              B2_preset<-as.numeric(myB1)

              B.scale=as.double(100)
              divide = get("/")
              S.init=divide(B2_preset,B.scale)  
      
      }
  
      if(p0==1)
      {
          postV=read.delim(v_name,header = FALSE)
          
          startindex=postV[1]
          startindex=(as.character(unlist(startindex)))
          startindex=as.numeric(startindex)
          
          v_index=match( iteration_index,startindex)
          
          postV=read.csv(v_name,header = FALSE)
          postV=(as.character(unlist(postV)))
          postV<-(strsplit(postV, " "))
          postV=postV[(v_index): (as.integer(v_index+2*k0-1))]
          postV<-split(postV,1:2)
          postV=unlist(postV[2])
      
      #mu.mdp
      postmu=read.csv(muname, header = FALSE)
      postmu=(as.character(unlist(postmu)))
      postmu1=postmu[mem_index]
      
      postmu1<-unlist(strsplit(postmu1, "\t"))
      postmu1<-postmu1[seq(3,length(postmu1),3)]
     }
          
        #mj1
      
      if(p1>0) {
        

        mymj1= read.csv2(mj1name,header = FALSE)
        
        mymj1=(as.character(unlist(mymj1)))
        mymj1<-(strsplit(mymj1, " "))
        nu_mj1=as.numeric(unlist(mymj1))
       
        nu_mj1=nu_mj1[!is.na(nu_mj1)]
        mj_index=match( iteration_index,nu_mj1)
        
        mymj1_1=nu_mj1[(mj_index+2): (mj_index+2+k0*p1*2-1)]
        m<-split(mymj1_1,1:2)
        mym1<-matrix(unlist(m[1]),ncol = p1, nrow =k0)
        
        mysj1<-matrix(unlist(m[2]),ncol = p1, nrow =k0)
      }
      if(p2>0)
      {
        
        #pij2
        mypij2= read.delim(pij2name,header = FALSE)
        mypij2=(as.character(unlist(mypij2)))
        mypij2<-(strsplit(mypij2, " "))
        mypij2=as.numeric(unlist(mypij2))
        mypij2=mypij2[!is.na(mypij2)]
        pij_index=match( iteration_index,mypij2)
        
        mypij2_2=mypij2[(pij_index+2): (pij_index+2+k0*p2*2-1)]
        mypij2_2=unlist(mypij2_2)
        
        
      }
        
         #aj4 and bj4
        if(p4>0)
        {
              myaj4= read.csv2(aj4name,header = FALSE)
              
              myaj4=(as.character(unlist(myaj4)))
              myaj4<-(strsplit(myaj4, " "))
              myaj4=as.numeric(unlist(myaj4))
              myaj4=myaj4[!is.na(myaj4)]
              
              aj_index=match( iteration_index,myaj4)
              
              myaj4=myaj4[(aj_index+2): (aj_index+2+k0*p4*2-1)]
              myaj4<-split(myaj4,1:2)
              aj4<-unlist(myaj4[1])
              bj4<-unlist(myaj4[2])
           }
      

        ###survival parameter p0=2
        if(p0>1)
        {
            exp_alpha_vec<-matrix(0,1,p0)
            exp_beta_vec<-matrix(0,1,p0)
            exp_alpha= read.csv2(survaname,header = FALSE)

            exp_alpha=(as.character(unlist(exp_alpha)))
            exp_alpha<-(strsplit(exp_alpha, " "))
            exp_alpha=as.numeric(unlist(exp_alpha))
            exp_alpha=exp_alpha[!is.na(exp_alpha)]
            alpha_index=match( iteration_index,exp_alpha)
            for(i in 1:p0){
              exp_alpha_vec[i]=exp_alpha[alpha_index+i]
            }

             exp_beta= read.csv2(survbname,header = FALSE)

            exp_beta=(as.character(unlist(exp_beta)))
            exp_beta<-(strsplit(exp_beta, " "))
            exp_beta=as.numeric(unlist(exp_beta))
            exp_beta=exp_beta[!is.na(exp_beta)]
            beta_index=match( iteration_index,exp_beta)
            for(i in 1:p0){
              exp_beta_vec[i]=exp_beta[beta_index+i]
            }

           
            lambda_all= read.delim(lambdaname,header = FALSE)
            indexcar=as.character(iteration_index)
            lambda_index=which(lambda_all==indexcar)
            lambda_all_data<-c(lambda_all[ (lambda_index+1 ): (lambda_index+k0 ),] )
            lambda_numeric <- matrix(as.numeric(unlist(strsplit(lambda_all_data, " "))), ncol = 2, byrow = TRUE)
   

            theta_all=read.delim(copulaname,header = FALSE)
            indextheta=as.character(iteration_index)
            theta_index=which(theta_all==indextheta)
            theta_all_data<-c(theta_all[ (theta_index+1 ): (theta_index+k0 ),] )
            theta_numeric <- c(as.numeric(unlist((theta_all_data))))

        }
      
      
    }
 
    member0<-as.numeric(as.factor(rank(member0)))
    member0<-member0-1
   

  
    if(p0>1)
    {

        g1<-optimize(f = loglike_lambda, Y[,1], interval = c(0,1), maximum = TRUE)
        g2<-optimize(f = loglike_lambda, Y[,2], interval = c(0,1), maximum = TRUE)

     

        g1<-unlist(g1,use.names = FALSE)[1]
        g2<-unlist(g2,use.names = FALSE)[1]
        g1_lmd <-as.double(g1)
        g2_lmd<-as.double(g2)
        g1_shape=0
        g2_shape=0
   
      
    }
  


    ## 3. write init file
    options(digits=2)
    out <- file("init.mdp",open="w")
 
    cat(" k0 ", k0, 
        "\n member0 ", member0, "\n", # dummy if k0=1 or n
        file=out)
    
    
    #edit
    if(p1>0  && postMCMC==1)
    {
      cat("\n sim-mj  ",mym1, "\n" ,file=out)
      cat("\n sim-sj1  ",mysj1, "\n" ,file=out)
      cat("\n sim-pij  ",mypij2_2,"\n" ,file=out)
      if(p4>0){
         cat("\n sim-aj4  ",aj4,"\n" ,file=out)
        cat("\n sim-bj4  ",bj4,"\n" ,file=out)
      }

      if(p0==1){
        cat("\n mdp-V ",postV,"\n",file=out)
        cat("\n mdp-mu1 ",postmu1,"\n",file=out)
      
      }
      else{
         cat("\n exp-alpha ",exp_alpha_vec,"\n",file=out)
         cat("\n exp-beta ",exp_beta_vec ,"\n",file=out)
         
        # cat("\n lambda-j ",lambda_numeric,"\n",file=out)
         
          cat("\n lambda-j ",  file=out)
              for (i in 1:k0) {
              for(j in 1:p0 ){
                cat(" ", lambda_numeric[i,j], file=out)
      
              }
            }
          cat("\n",file=out)
          cat("\n theta ",theta_numeric,"\n",file=out)
      }
      
     
    }

    close(out)
    if(censoring && p0==1)
    {
        col_cenindex=p0+p1+p2+p4+1
        
        Y<-Y %>% relocate(col_cenindex,.before=p0+1)
 
    }
    else if(censoring && p0 >1)
    {
       
       col_cenindex=p0+p1+p2+p4+1
        
       Y<-Y %>% relocate(col_cenindex,.before=p0+1)
       Y<-Y %>% relocate(col_cenindex+1,.before=p0+2)
    }
    
    


    ## 4. write data file
    out <- file("data.mdp",open="w")
    write((t(Y)),ncolumns=p,file=out)
    close(out)
    
    
    initname <- paste("py-init_", seeds, ".mdp", sep = "")

    ## 5. write py-init file
    if (py==1){
      out <- file(initname,open="w")
      cat("\n ny ",ny, "nx ",nx, "yi ", iy,
          "\n ygrid ", ygrid,
           "\n ygrid2 ", ygrid2,
          "\n xgrid \n",
          file=out)
      write(format(t(xgrid)),ncolumns=ncol(xgrid),file=out)
      close(out)
    }


    
    names(q2)<-NULL
    n<-as.double(n)
    postMCMC<-as.double(postMCMC)
     p<-as.double(p)
    p0<-as.double(p0)
    censoring<-as.double(censoring)
    p1<-as.double(p1)
    p2<-as.double(p2)
    q2<-as.double(q2)
    print(q2)
    p4<-as.double(p4)
    cov.dahl<-as.double(cov.dahl)
    if(p1>0)
    {
        cov.Vj1 <- as.double(cov.Vj1)
      cov.mean1<-as.double(cov.mean1)
      cov.B1<-as.double(cov.B1)
    }
    else{
      cov.Vj1 <- 0
      cov.mean1<- 0
      cov.B1<- 0

    }
   

    cov.alpha1 <-as.double(rep(1.0,p1))
    cov.pi2
    cov.a4<-as.double(cov.a4)
    cov_b4<-as.double(cov.b4)
    cov.alpha4<-as.double(rep(1.0,p4))
    n.iter<-as.double(n.iter)

    m.prior<-as.double(m.prior)
    B.prior<-as.double(B.prior)
     S.prior<-as.double(S.prior)
     n.discard<-as.double(n.discard) 
    n.reinit<-as.double(n.reinit)
     n.batch<- as.double(n.batch)
       n.predupdate<-as.double(n.predupdate)
       n.printallpars<-as.double(n.printallpars)
        verbose<-as.double(verbose)
         py<-as.double(py)
        s<-as.double(s)
       S.init<-as.double(S.init)
        q<-as.double(qq)
      simu<-as.double(simu)
      tr_p1<-as.double(tr_p1)
      tr_p2<-as.double(tr_p2)
     
       if(p0>1)
       {

       
         R.inv <-solve(R)
         names(R.inv)<-NULL
         # B-scale<-B.scale
        c <-cc
        m.init <-as.double(m.init)
         aa <-as.double(a)
         names(A.inv)<-NULL
         #A-inv<-(A.inv
       }
      else{
        R <-as.double(R)
        B.scale <-as.double(B.scale)
        c <-as.double(cc)
       m.init <-as.double(m.init)
        aa<-as.double(a)
        A <-(1/A.inv)
        names(A)<-NULL
    }
        
        alpha.init <-as.double( alpha)
        a0 <-as.double(  a0) 
        b0 <-as.double(  b0)
  
        sigma1<-as.double(sigma1)
        sigma2<-as.double(sigma2)
        seeds<-as.double(seeds)
     
     
    if(p0>1){
        init<-list("n"=n,"postMCMC"=postMCMC,"p"=p,"p0"=p0, "censoring"=censoring,"p1"=p1,"p2"=p2,"q2"=q2,"p4"=p4,"cov.dahl"=cov.dahl, 
               "cov.Vj1 "=cov.Vj1,"cov.mean1"=cov.mean1,"cov.B1"=cov.B1, "cov.alpha1"=cov.alpha1,"cov.a4"=cov.a4,
               "cov.b4"=cov_b4,"cov.alpha4"=cov.alpha4,"n.iter"=n.iter,"m.prior"=m.prior,"B.prior"=B.prior,"S.prior"=S.prior,"n.discard"=n.discard,
               "n.reinit"=n.reinit,"n.batch"=n.batch," n.predupdate"= n.predupdate,"n.printallpars"=n.printallpars,"verbose"=verbose,"py"=py,
               "s"=s,"S.init"=S.init,"q"=q,"R.inv"=R.inv,"B.scale"=B.scale,"c"=c,"m.init"=m.init,"aa"=aa,"A.inv"=A.inv,"alpha.init"=alpha.init,
               "a0"=a0,"b0"=b0,"g1_lmd"=g1_lmd, "g2_lmd"=g2_lmd,"g1_shape"=g1_shape,"g2_shape"=g2_shape, "sigma_1"= sigma1,"sigma_2"=sigma2,"seeds"=seeds ,"simu"=simu,"tr_p1"=tr_p1,"tr_p2"=tr_p2,"cov.pi2"=cov.pi2) 
               ##### cov.pi2 must be the last element in the list
    }
    else{
      init<-list("n"=n,"postMCMC"=postMCMC,"p"=p,"p0"=p0, "censoring"=censoring,"p1"=p1,"p2"=p2,"q2"=q2,"p4"=p4,"cov.dahl"=cov.dahl,
               "cov.Vj1 "=cov.Vj1,"cov.mean1"=cov.mean1,"cov.B1"=cov.B1, "cov.alpha1"=cov.alpha1,"cov.a4"=cov.a4,
               "cov.b4"=cov_b4,"cov.alpha4"=cov.alpha4,"n.iter"=n.iter,"m.prior"=m.prior,"B.prior"=B.prior,"S.prior"=S.prior,"n.discard"=n.discard,
               "n.reinit"=n.reinit,"n.batch"=n.batch," n.predupdate"= n.predupdate,"n.printallpars"=n.printallpars,"verbose"=verbose,"py"=py,
               "s"=s,"S.init"=S.init,"q"=q,"R"=R,"B.scale"=B.scale,"c"=c,"m.init"=m.init,"aa"=aa,"A"=A,"alpha.init"=alpha.init,
               "a0"=a0,"b0"=b0,"cov.pi2"=cov.pi2)

    }
    
    
    
    return(init)

  }


## auxilary functions
n.unique <- function(x)
  { # returns number of unique values
    length(unique(x))
  }

cross.tab <-  function(Y2,q2)
{ ## computes cross tabulation of categorical var's in the columns of Y2,
  ## into k0 distinct combinations x*[j], j=0..k0-1, and
  ## returns vector of indicators s0[i]=j iff X[i]=x*[j]
  ## q2[j]=number of distinct values of j-th categorical var
  p2 <- length(q2)
  
  Y2 <- t( t(Y2) - apply(Y2,2,min)) # now each categorical starts at 0
  s0 <- Y2[,1]
  offset <- q2[1]
  for(j in 2:p2){
    s0 <- s0+Y2[,j]*offset
    offset <- q2[j]*offset
  }
  return(s0)
}

cross.tab2 <- function(q12,nx,p1)
  { ## returns list of nx design vectors
    ## with all possible cominations of the p12 variables with
    ## q12[j]= n levels of j-th variable
    ## (to be used with q12[j]=3 for continous covariates to return
    ## -1,0,1 for lo, med, hi

    p12 <- length(q12)
    xgrid <- matrix(0,nrow=nx,ncol=p12)
    for(i in 1:nx){
      x <- i-1
      for(j in 1:p12){
        xgrid[i,j] <- x %% q12[j]
        x <-  x %/% q12[j]
      }
    }
    if(p1>0)
      xgrid[,1:p1] <- xgrid[,1:p1]-1 # to code -1,0,1
    return(xgrid)
}

find_freq <- function(x) {
  ux <- unique(x)
  ux[which.max(tabulate(match(x, ux)))]
}

loglike_lambda <- function(lambda, x){
  length(x) * log(lambda) - lambda*length(x)*mean(x)
}

